% This code runs a simulation for coverage probabilities of bootstrap CIs
% For the detail, see the online appendix of Han and Lee (2019) 

% Semiparametric Model : Transformed model 
% Choice of G: Standard normal 
% Parametric Specification : Normal 
% True Marginal Distribution : Normal 


clear;
clc;

%% Design of Simulation : Parameters

n = 500 ; % Number of observations
m = 2; % Number of simulation iterations
bb = 5; % Number of bootstrappings %

kne = floor(3* n^(1/7)) ; % Number of Polynomials for epsilon 
knv = floor(3* n^(1/7)) ;

cop = [0;2; 3; 5];

cop_ind = 1 ;

cop_opt = cop(cop_ind,1) ;    % Gaussian = 0, Frank = 2, Clayton = 3, Gumbel = 5
cop_opt_DGP = cop(cop_ind,1);

beta1 = -1 ;
beta2 = 0.8  ;

alpha1 = -1 ;       
alpha2 = 0.5;

beta = [beta1; beta2]; 
alpha = [alpha1; alpha2];

dx = size(beta, 1) ;
    

delta = 1.1;
gamma = 0.8 ;

dz = size(gamma,1) ;

beta_base = beta(1) ;
alpha_base = alpha(1) ;

%% DGP : Marginal Distributions and Covariates

mu_xz = zeros(dx+dz,1);
mu_x = mu_xz(1:dx,1) ;
mu_z = mu_xz(dx+1:dx+dz) ;

corr12 = -0.2 ;
corr1z = -0.1 ;
corr2z = 0.1 ;

if dx == 1 
    sigma_xz = [1, corr1z; corr1z, 1];
    
elseif dx == 2
    sigma_xz = [1, corr12, corr1z ; corr12, 1, corr2z; corr1z, corr2z, 1];
    
end

 
mu_e = 0  ;
mu_v = 0 ;

sigma_e = sqrt(1) ;
sigma_v = sqrt(1) ;

nuisance = [mu_e; mu_v; sigma_e; sigma_v] ;

ate = normcdf(mu_x'*beta + delta, mu_e, sigma_e) - normcdf(mu_x'*beta, mu_e, sigma_e) ;

%% DGPs : Copula


% Normal copula
if cop_opt_DGP == 0
    rho = 0.5176;  % sp_rho = 0.5, NO LARGER THAN 0.9!!
    sp_rho = copulastat('Gaussian',rho,'type','Spearman') ;
end
% Example 3.2: The Frank family
if cop_opt_DGP == 2
    rho = 3.446;    % sp_rho = 0.5  
    sp_rho = copulastat('Frank',rho,'type','Spearman');
end    
% Example 3.3: The Clayton family (The Kimeldorf and Sampson family, in Joe)
if cop_opt_DGP == 3
    rho = 1.076;    % sp_rho = 0.5
    sp_rho = copulastat('Clayton',rho,'type','Spearman');
end
% Example 3.5: The Gumbel family
if cop_opt_DGP == 5
    rho = 1.541;    % sp_rho = 0.5     
    sp_rho = copulastat('Gumbel',rho,'type','Spearman');
end

%% Specifications : Copula

% Normal copula
if cop_opt == 0
    rho_mid = 0.5 ;
    rho_L = -0.9;
    rho_H = 0.9;
end
% Example 3.2: The Frank family
if cop_opt == 2
    rho_mid = 3.446;    % sp_rho = 0.5
    rho_L = rho_mid - 2;   % lower bound is infinity
    rho_H = rho_mid + 2;    % upper bound is infinity
    %EXCEPT rho = 0;
end    
% Example 3.3: The Clayton family (The Kimeldorf and Sampson family, in Joe)
if cop_opt == 3
    rho_mid = 1.076;    % sp_rho = 0.5
    rho_L = 0 + 0.01;   % lower bound is -1, but somehow error occurs...
    rho_H = rho_mid + 2;    % upper bound is infinity
    %EXCEPT rho = 0;
end% Example 3.5: The Gumbel family
if cop_opt == 5
    rho_mid = 1.541;    % sp_rho = 0.5
    rho_L = 1 + 0.01;   % lower bound is 1
    rho_H = rho_mid + 2;    % upper bound is infinity
end



%% Option 
options = optimset('LargeScale','off', ...
               'HessUpdate','bfgs', ...
               'Algorithm', 'active-set',...
               'Hessian','off', ...
               'GradObj','off', ...
               'DerivativeCheck','off',...
               'Display', 'off');



%% Matrice for Results 

for mm = 1 : m 

theta = [alpha; gamma; beta; delta; rho] ;

kk = size(theta,1) ; % Number of finite-dimensional parameters

theta_ext = [theta; sp_rho] ;


%% Estimation

if cop_opt == 0
    copula_hat = @(u1,u2,rho)copulacdf('Gaussian',[u1,u2],[1,rho;rho,1]);
    %copula = @(u1,u2,rho)normcop(u1,u2,rho);
elseif cop_opt == 2
    copula_hat = @(u1,u2,rho)copulacdf('Frank',[u1,u2],rho);
elseif cop_opt == 3
    copula_hat = @(u1,u2,rho)copulacdf('Clayton',[u1,u2],rho);
elseif cop_opt == 5
    copula_hat = @(u1,u2,rho)copulacdf('Gumbel',[u1,u2],rho);
end


%Initial values 
theta_0 = theta + ((-1 + 2*rand(kk,1))/10) ; 
theta_0(1) = alpha_base ;
theta_0(dx+dz+1) = beta_base;
theta_0(end) = rho_mid ;

xie_0 = zeros(kne,1) ;
xiv_0 = zeros(knv,1) ;
cons_0 = 1 ;

semi_par_0 = [theta_0 ; xie_0; cons_0; xiv_0; cons_0] ;

mu_0 = 0 ;
sigma_0 = 1 ;

norm_par_0 = [theta_0 ; mu_0; mu_0; sigma_0; sigma_0] ;

if cop_opt_DGP == 0
    sigma_u =[1,rho;
        rho,1];
    u = copularnd('Gaussian',sigma_u,n);
end

if cop_opt_DGP == 2
    u = copularnd('Frank',rho,n);
    %u = copula2rnd(n, rho);
end

if cop_opt_DGP == 3
    u = copularnd('Clayton',rho,n);
end

if cop_opt_DGP == 5
    u = copularnd('Gumbel',rho,n);
    %u = copula5rnd(n, rho);

end

u1 = u(:,1);
u2 = u(:,2);

%x1 = normrnd(0,1,n,1)*sigma_x1 + mu_x1 ;
%x2 = normrnd(0,1,n,1)*sigma_x2 + mu_x2 ;

xz = mvnrnd(mu_xz, sigma_xz, n) ;

x = xz(:,1:dx);
z = xz(:,(dx+1):(dx+dz)) ;

x_bar = mean(x) ;

%if dx == 1 
%    x = x1 ;
%else
%    x = [x1, x2];
%end

%z = normrnd(0,1,n,1)*sigma_z + mu_z ;

d = (normcdf((x*alpha + z*gamma - mu_v)./sigma_v, 0, 1) >= u1);  
y = (normcdf((x*beta + delta*d - mu_e)./sigma_e, 0, 1) >= u2);   


data = [y,d,x,z];

[semi_par_est, semi_logL_val] = fmincon(@(par)semi_logl(data, par, cop_opt, dx, dz, kne, knv),semi_par_0, [], [], [], [], [], [], @(par)semi_constraint(par, dx, dz, kk, kne, knv, beta_base, alpha_base), options);

semi_theta_est_all = semi_par_est(1:kk) ;
semi_beta_est = semi_par_est(dx+dz+1:dx+dz+dx);
semi_delta_est = semi_par_est(dx+dz+dx+1) ;
xie_est_all = semi_par_est(kk+1:kk+kne) ;
conse_est_all = semi_par_est(kk+kne+1) ;
xiv_est_all = semi_par_est(kk+kne+2:kk+kne+1+knv) ;
consv_est_all = semi_par_est(kk+kne+knv+2) ;

p_dim = size(semi_par_est, 1) ;


semi_ate_est_all = dist(normcdf(x_bar*semi_beta_est + semi_delta_est,0,1), semi_par_est(kk+1:kk+kne+1), kne) - dist(normcdf(x_bar*semi_beta_est,0,1), semi_par_est(kk+1:kk+kne+1), kne);


if cop_opt == 0 
    semi_sp_rho_est_all = copulastat('Gaussian',semi_par_est(kk),'type','Spearman');

elseif cop_opt == 2 
    semi_sp_rho_est_all = copulastat('Frank',semi_par_est(kk),'type','Spearman');

elseif cop_opt == 3 
    semi_sp_rho_est_all = copulastat('Clayton',semi_par_est(kk),'type','Spearman');

elseif cop_opt == 5
    semi_sp_rho_est_all = copulastat('Gumbel',semi_par_est(kk),'type','Spearman');

end


%%%%%% Bootstrap %%%%%

boot_semi_est = zeros(bb, p_dim) ;
boot_semi_ate = zeros(bb,1) ;



parfor j = 1 : bb 
    
    weight = exprnd(1,n,1) ;   
        
                
    [boot_semi_par_est, boot_semi_logL_val] = fmincon(@(par)boot_semi_logl(data, par, cop_opt, dx, dz, kne, knv, weight),semi_par_est, [], [], [], [], [], [], @(par)semi_constraint(par, dx, dz, kk, kne, knv, beta_base, alpha_base), options);
    
    boot_semi_beta_est = boot_semi_par_est(dx+dz+1:dx+dz+dx);
    boot_semi_delta_est = boot_semi_par_est(dx+dz+dx+1) ;
    
    boot_semi_est(j,:) = boot_semi_par_est' ;
    boot_semi_ate(j,1) = dist(normcdf(x_bar*boot_semi_beta_est + boot_semi_delta_est,0,1), boot_semi_par_est(kk+1:kk+kne+1), kne) - dist(normcdf(x_bar*boot_semi_beta_est,0,1), boot_semi_par_est(kk+1:kk+kne+1), kne) ;
    
    
end



save(sprintf('Test_Semi_Bootstrap_%d_%d_%d_%d_k%d_%d.mat',n,bb,cop_opt_DGP,cop_opt,kne, mm),'semi_theta_est_all', 'xie_est_all', 'xiv_est_all', 'conse_est_all', 'consv_est_all', 'semi_ate_est_all', 'p_dim', 'boot_semi_est', 'boot_semi_ate', 'theta', 'ate', 'kk', 'n', 'm', 'bb') ;
end
